NAVDatabase.prototype.adminCommand = function (command) {
    if (this.getName() === "admin")
        return this.runCommand(command);
    return this.getSiblingDB("admin").runCommand(command);
};

NAVDatabase.prototype.aggregate = function (pipeline, options) {
    var tmpPipeline = undefined;
    var tmpOptions = undefined;
    if (!Array.isArray(pipeline)) {
        tmpPipeline = Array.from(arguments);
        tmpOptions = {};
    } else {
        tmpPipeline = Object.extend([], (pipeline || []));
        tmpOptions = Object.extend({}, (options || {}));
    }

    if (!Array.isArray(tmpPipeline))
        nav_throwError("pipeline must be an array");

    if (isUndefined(tmpOptions))
        tmpOptions = {};

    if (!isObject(tmpOptions))
        nav_throwError("options must be an object");

    if (!("cursor" in tmpOptions))
        tmpOptions.cursor = {};

    var maxAwaitTimeMS = tmpOptions.maxAwaitTimeMS;
    delete tmpOptions.maxAwaitTimeMS;

    var commandObj = { aggregate: 1, pipeline: pipeline };
    Object.extend(commandObj, tmpOptions);

    var hasOutStage = pipeline.length >= 1 && pipeline[pipeline.length - 1].hasOwnProperty("$out");
    
    var isWatch = false;
    if (pipeline.length > 0)
        isWatch = pipeline[0].$changeStream ? true : false;

    return this.forwardToCustomFunction("databaseAggregate", hasOutStage, commandObj, tmpOptions.explain, maxAwaitTimeMS, isWatch);
};

NAVDatabase.prototype.cloneCollection = function (from, collection, query) {
    collection = this.getName() + "." + collection;
    query = query || {};
    var commandObj = { cloneCollection: collection, from: from, query: query };
    return this.runCommand(commandObj);
};

NAVDatabase.prototype.cloneDatabase = function (hostName) {
    var commandObj = { clone: hostName };
    return this.runCommand(commandObj);
};

NAVDatabase.prototype.commandHelp = function (command) {
    if (isUndefined(command))
        nav_throwError("CommandHelp should have command");
    if (!isString(command))
        nav_throwError("Command should be string");

    var commandObj = { [command]: 1, help: true };
    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);
    return result.help;
};

NAVDatabase.prototype.copyDatabase = function (fromDB, toDB, fromHost, userName, password, mechanism, slaveOk) {
    if (!isUndefined(fromHost) || !isUndefined(userName) || !isUndefined(password) || !isUndefined(mechanism) || !isUndefined(slaveOk))
        nav_throwError("copyDatabase between different servers is not supported");

    var commandObj = { copydb: 1 };
    if (!isUndefined(fromDB))
        commandObj.fromdb = fromDB;
    if (!isUndefined(toDB))
        commandObj.todb = toDB;

    return this.adminCommand(commandObj);
};

NAVDatabase.prototype.createCollection = function (collectionName, options) {
    var tmpOptions = Object.extend({}, (options || {}));

    var commandObj = { create: collectionName };
    Object.extend(commandObj, tmpOptions);
    return this.runCommand(commandObj);
};

NAVDatabase.prototype.createView = function (viewName, source, pipeline, options) {
    var tmpOptions = Object.extend({}, (options || {}));

    var commandObj = { create: viewName };

    if (isUndefined(source))
        nav_throwError("Must specify a view or collection");
    if (!isString(source))
        nav_throwError("View or collection name should be string");

    if (!isUndefined(pipeline)) {
        if (!Array.isArray(pipeline))
            pipeline = [pipeline];
    }
    tmpOptions.pipeline = pipeline;
    tmpOptions.viewOn = source;

    Object.extend(commandObj, tmpOptions);

    return this.runCommand(commandObj);
};

NAVDatabase.prototype.currentOp = function (operations) {

    var options = {};
    if (operations) {
        if (isObject(operations))
            Object.extend(options, operations);
        else if (operations)
            options["$all"] = true;
    }

    var pipeline = [];

    var currentOpObj = {};
    currentOpObj["allUsers"] = !options["$ownOps"];
    currentOpObj["idleConnections"] = !!options["$all"];

    // documentDB does not support truncateOps flag
    if (!this.getMongo().forwardToCustomFunction("mongoCCGetValue", "isMongoDocumentDB"))
        currentOpObj["truncateOps"] = false;

    pipeline.push({ "$currentOp": currentOpObj });

    var matchObj = {};
    for (const fieldname of Object.keys(options)) {
        if (fieldname !== "$all" && fieldname !== "$ownOps" && fieldname !== "$truncateOps") {
            matchObj[fieldname] = options[fieldname];
        }
    }

    pipeline.push({ "$match": matchObj });

    var results = this.getSiblingDB("admin").aggregate(pipeline,
        { "$readPreference": { "mode": "primaryPreferred" } }).toArray();
        
    return { "inprog": results.length > 0 ? results : [], "ok": 1 };
};
NAVDatabase.prototype.currentOP = NAVDatabase.prototype.currentOp;

NAVDatabase.prototype.dropDatabase = function (writeConcern) {
    var commandObj = { dropDatabase: 1 };
    if (writeConcern)
        commandObj.writeConcern = writeConcern;
    return this.runCommand(commandObj);
};

NAVDatabase.prototype.eval = function (fucntinoScript) {
    var commandObj = { $eval: fucntinoScript };
    if (arguments.length > 1)
        commandObj.args = Array.from(arguments).slice(1);

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result.retval;
};
NAVDatabase.prototype.dbEval = NAVDatabase.prototype.eval;

NAVDatabase.prototype.fsyncLock = function () {
    var commandObj = { fsync: 1, lock: true };
    return this.adminCommand(commandObj);
};

NAVDatabase.prototype.fsyncUnlock = function () {
    return this.forwardToCustomFunction("databaseFsyncUnlock");
};

NAVDatabase.prototype.nav_fsyncUnlock = function () {
    return this.getSiblingDB('admin').$cmd.sys.unlock.findOne();
};

NAVDatabase.prototype.getCollection = function (collectionName) {
    if (isUndefined(collectionName))
        nav_throwError("getCollection should have argument");
    if (!isString(collectionName))
        nav_throwError("Collection name should be string");
    return this.forwardToCustomFunction("databaseGetCollection", collectionName);
};

NAVDatabase.prototype._getCollectionInfosCommand = function (filter, nameOnly, authorizedCollections) {
    filter = filter || {};
    var commandObj = { 
                       listCollections: 1,
                       filter: filter,
                       nameOnly: nameOnly ? true : false,
                       authorizedCollections: authorizedCollections ? true : false
                     };
    var result = this.runCommand(commandObj);
    if (result.code == 59)
        return null;

    if (!result.ok) {
        if (result.errmsg && result.errmsg.startsWith("no such cmd"))
            return null;
        nav_throwError(result.errmsg, result);
    }

    return result.cursor.firstBatch.sort(compareOn("name"));
};

NAVDatabase.prototype._getCollectionInfosSystemNamespaces = function (filter) {
    var collectionList = [];

    var databaseNamePrefix = this.getName() + ".";

    filter = filter || {};
    if (isString(filter.name))
        filter.name = databaseNamePrefix + filter.name;

    var result = this.getCollection("system.namespaces").find(filter);
    while (result.hasNext()) {
        var collectionObj = result.next();

        if (collectionObj.name.indexOf("$") >= 0 && collectionObj.name.indexOf(".oplog.$") < 0)
            continue;

        collectionObj.name = collectionObj.name.substring(databaseNamePrefix.length);
        collectionList.push(collectionObj);
    }

    return collectionList.sort(function (collection1, collection2) {
        return collection1.name.localeCompare(collection2.name);
    });
};

NAVDatabase.prototype.getCollectionInfos = function (filter, nameOnly, authorizedCollections) {
    var version = this.getDatabaseVersion();
    if (version >= 30000)
        return this._getCollectionInfosCommand(filter, nameOnly ? true : false, authorizedCollections ? true : false);
    else
        return this._getCollectionInfosSystemNamespaces(filter);
};

NAVDatabase.prototype.getCollectionNames = function () {
    return this.forwardToCustomFunction("databaseGetCollectionNames");
};

NAVDatabase.prototype.getDatabaseVersion = function () {
    return this.forwardToCustomFunction("databaseGetDatabaseVersion");
};

NAVDatabase.prototype.getLastError = function (w, wtimeout) {
    var result = this.getLastErrorObj(w, wtimeout);
    if (!result.ok)
        nav_throwError(result.errmsg, result);
    return result.err;
};

NAVDatabase.prototype.getLastErrorObj = function (w, wtimeout, j) {
    var commandObj = { getlasterror: 1 };
    if (w) {
        commandObj.w = w;
        if (wtimeout)
            commandObj.wtimeout = wtimeout;
        if (j != null)
            commandObj.j = j;
    }

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);
    return result;
};
NAVDatabase.prototype.getLastErrorCmd = NAVDatabase.prototype.getLastErrorObj;

NAVDatabase.prototype.getLogComponents = function () {
    var commandObj = { getParameter: 1, logComponentVerbosity: 1 }

    var result = this.adminCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);
    return result.logComponentVerbosity;
};

NAVDatabase.prototype.getMongo = function () {
    return this.forwardToCustomFunction("databaseGetMongo");
};

NAVDatabase.prototype.getName = function () {
    return this.forwardToCustomFunction("databaseGetName");
};

NAVDatabase.prototype.getPrevError = function () {
    var commandObj = { getpreverror: 1 };
    return this.runCommand(commandObj);
};

NAVDatabase.prototype.getProfilingLevel = function () {
    var commandObj = { profile: -1 };
    var result = this.runCommand(commandObj);
    return result ? result.was : null;
};

NAVDatabase.prototype.getProfilingStatus = function () {
    var commandObj = { profile: -1 };
    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);
    delete result.ok;
    return result;
};

NAVDatabase.tsToSeconds = function (ts) {
    if (ts.t && ts.i)
        return ts.t;
    return ts / 4294967296;
};

NAVDatabase.prototype.getReplicationInfo = function () {
    var localdb = this.getSiblingDB("local");

    var result = {};
    var oplog;
    var localCollections = localdb.getCollectionNames();
    if (localCollections.indexOf('oplog.rs') >= 0) {
        oplog = 'oplog.rs';
    } else if (localCollections.indexOf('oplog.$main') >= 0) {
        oplog = 'oplog.$main';
    } else {
        result.errmsg = "neither master/slave nor replica set replication detected";
        return result;
    }

    var ol = localdb.getCollection(oplog);
    var ol_stats = ol.stats();
    if (ol_stats && ol_stats.maxSize) {
        result.logSizeMB = ol_stats.maxSize / (1024 * 1024);
    } else {
        result.errmsg = "Could not get stats for local." + oplog + " collection. " +
            "collstats returned: " + tojson(ol_stats);
        return result;
    }

    result.usedMB = ol_stats.size / (1024 * 1024);
    result.usedMB = Math.ceil(result.usedMB * 100) / 100;

    var firstCursor = ol.find().sort({ $natural: 1 }).limit(1);
    var lastCursor = ol.find().sort({ $natural: -1 }).limit(1);
    if (!firstCursor.hasNext() || !lastCursor.hasNext()) {
        result.errmsg = "objects not found in local.oplog.$main -- is this a new and empty db instance?";
        result.oplogMainRowCount = ol.count();
        return result;
    }

    var firstObj = firstCursor.next();
    var lastObj = lastCursor.next();
    var tfirst = firstObj.ts;
    var tlast = lastObj.ts;

    if (tfirst && tlast) {
        tfirst = NAVDatabase.tsToSeconds(tfirst);
        tlast = NAVDatabase.tsToSeconds(tlast);
        result.timeDiff = tlast - tfirst;
        result.timeDiffHours = Math.round(result.timeDiff / 36) / 100;
        result.tFirst = (new Date(tfirst * 1000)).toString();
        result.tLast = (new Date(tlast * 1000)).toString();
        result.now = Date();
    } else {
        result.errmsg = "ts element not found in oplog objects";
    }

    return result;
};

NAVDatabase.prototype.getSiblingDB = function (databaseName) {
    if (isUndefined(databaseName))
        nav_throwError("getSiblingDB should have argument");
    if (!isString(databaseName))
        nav_throwError("Database name should be string");
    return this.forwardToCustomFunction("databaseGetSiblingDB", databaseName);
};
NAVDatabase.prototype.getSisterDB = NAVDatabase.prototype.getSiblingDB;

NAVDatabase.prototype.hello = function () {

    var version = this.getDatabaseVersion();
    if (version >= 50000) {
        var commandObj = { hello: 1 };
        return this.runCommand(commandObj);
    } else {
        var result = this.isMaster();
        delete result.ismaster;
        return result;
    }
};

NAVDatabase.prototype.help = function () {
    print("DB methods:\n" +
                     "\tdb.adminCommand(nameOrDocument) - switches to 'admin' db, and runs command [just calls db.runCommand(...)]\n" +
                     "\tdb.aggregate([pipeline], {options}) - performs a collectionless aggregation on this database; returns a cursor\n" +
                     "\tdb.auth(username, password)\n" +
                     "\tdb.cloneDatabase(fromhost)\n" +
                     "\tdb.commandHelp(name) returns the help for the command\n" +
                     "\tdb.copyDatabase(fromdb, todb, fromhost)\n" +
                     "\tdb.createCollection(name, {size: ..., capped: ..., max: ...})\n" +
                     "\tdb.createView(name, viewOn, [{$operator: {...}}, ...], {viewOptions})\n" +
                     "\tdb.createUser(userDocument)\n" +
                     "\tdb.currentOp() displays currently executing operations in the db\n" +
                     "\tdb.dropDatabase()\n" +
                     "\tdb.eval() - deprecated\n" +
                     "\tdb.fsyncLock() flush data to disk and lock server for backups\n" +
                     "\tdb.fsyncUnlock() unlocks server following a db.fsyncLock()\n" +
                     "\tdb.getCollection(cname) same as db['cname'] or db.cname\n" +
                     "\tdb.getCollectionInfos([filter]) - returns a list that contains the names and options of the db's collections\n" +
                     "\tdb.getCollectionNames()\n" +
                     "\tdb.getLastError() - just returns the err msg string\n" +
                     "\tdb.getLastErrorObj() - return full status object\n" +
                     "\tdb.getLogComponents()\n" +
                     "\tdb.getMongo() get the server connection object\n" +
                     "\tdb.getMongo().setSlaveOk() allow queries on a replication slave server\n" +
                     "\tdb.getName()\n" +
                     "\tdb.getPrevError()\n" +
                     "\tdb.getProfilingLevel() - deprecated\n" +
                     "\tdb.getProfilingStatus() - returns if profiling is on and slow threshold\n" +
                     "\tdb.getReplicationInfo()\n" +
                     "\tdb.getSiblingDB(name) get the db at the same server as this one\n" +
                     "\tdb.hostInfo() get details about the server's host\n" +
                     "\tdb.isMaster() check replica primary status\n" +
                     "\tdb.hello() check replica primary status\n" +
                     "\tdb.killOp(opid) kills the current operation in the db\n" +
                     "\tdb.listCommands() lists all the db commands\n" +
                     "\tdb.loadServerScripts() loads all the scripts in db.system.js\n" +
                     "\tdb.printCollectionStats()\n" +
                     "\tdb.printReplicationInfo()\n" +
                     "\tdb.printShardingStatus()\n" +
                     "\tdb.printSlaveReplicationInfo()\n" +
                     "\tdb.rotateCertificates(message) - rotates certificates, CRLs, and CA files and logs an optional message\n" +
                     "\tdb.dropUser(username)\n" +
                     "\tdb.repairDatabase()\n" +
                     "\tdb.resetError()\n" +
                     "\tdb.runCommand(cmdObj) run a database command.  if cmdObj is a string, turns it into {cmdObj: 1}\n" +
                     "\tdb.serverStatus()\n" +
                     "\tdb.setLogLevel(level,<component>)\n" +
                     "\tdb.setProfilingLevel(level,slowms) 0=off 1=slow 2=all\n" +
                     "\tdb.stats()\n" +
                     "\tdb.version() current version of the server\n");
};

NAVDatabase.prototype.hostInfo = function () {
    var commandObj = { hostInfo: 1 };
    return this.adminCommand(commandObj);
};

NAVDatabase.prototype.isMaster = function () {
    var commandObj = { isMaster: 1 };
    return this.runCommand(commandObj);
};

NAVDatabase.prototype.killOp = function (operation) {
    if (isUndefined(operation))
        nav_throwError("killOp should have argument");
    var commandObj = { killOp: 1, op: operation };
    return this.forwardToCustomFunction("databaseKillOp", commandObj, operation);
};
NAVDatabase.prototype.killOP = NAVDatabase.prototype.killOp;

NAVDatabase.prototype.nav_killOp = function (operationNumber) {
    return this.getSiblingDB("admin").$cmd.sys.killop.findOne({ op: operationNumber });
};

NAVDatabase.prototype.listCommands = function () {
    var commandObj = { listCommands: 1 };
    var result = this.runCommand(commandObj);

    var resultStr = "";
    for (var CommandName in result.commands) {
        var obj = result.commands[CommandName];

        var tmpStr = CommandName + ": ";

        if (obj.adminOnly)
            tmpStr += " adminOnly ";
        if (obj.slaveOk)
            tmpStr += " slaveOk ";
        if (obj.secondaryOk)
            tmpStr += " secondaryOk ";

        tmpStr += "\n  ";
        tmpStr += obj.help.replace(/\n/g, '\n  ');
        tmpStr += "\n\n";

        resultStr += tmpStr;
    }
    if (resultStr !== "")
        print(resultStr);
};

NAVDatabase.prototype.loadServerScripts = function () {
    var global = Function('return this')();
    this.system.js.find().forEach(function (functionScript) {
        if (functionScript.value.constructor === Code)
            global[functionScript._id] = eval("(" + functionScript.value.code + ")");
        else
            global[functionScript._id] = functionScript.value;
    });
};

NAVDatabase.prototype.printCollectionStats = function (scale) {
    if (!isUndefined(scale)) {
        if (!isNumber(scale))
            nav_throwError("scale should be a number >= 1");
        if (scale < 1)
            nav_throwError("scale should be >= 1");
    }
    var currentdb = this;
    var result = "";
    this.getCollectionNames().forEach(function (collectionName) {
        result += collectionName + "\n";
        result += tojson(currentdb.getCollection(collectionName).stats(scale)) + "\n";
        result += "---\n";
    });

    print(result);
};

NAVDatabase.prototype.printReplicationInfo = function () {
    var result = this.getReplicationInfo();
    var info = "";
    if (result.errmsg) {
        var isMaster = this.isMaster();
        if (isMaster.arbiterOnly) {
            info += "cannot provide replication status from an arbiter.\n";
        } else if (!isMaster.ismaster) {
            info += "this is a slave, printing slave replication info.\n";
            info += this.printSlaveReplicationInfo();
        } else
            info = tojson(result);
    } else {
        info += "configured oplog size:   " + result.logSizeMB + "MB\n";
        info += "log length start to end: " + result.timeDiff + "secs (" + result.timeDiffHours +
              "hrs)\n";
        info += "oplog first event time:  " + result.tFirst + "\n";
        info += "oplog last event time:   " + result.tLast + "\n";
        info += "now:                     " + result.now + "\n";
    }

    print(info);
};

NAVDatabase.prototype.printShardingStatus = function (verbose) {
    var configDB = this.getSiblingDB("config")

    var version = configDB.getCollection("version").findOne();
    if (version == null) {
        print("printShardingStatus: this db does not enable sharding . be sure you are connecting to a mongos from the shell and not to a mongod.");
        return;
    }

    var resultStr = "";
    var output = function (indent, str) {
        if (indent == 0)
            indent = 0;
        else if (indent == 1)
            indent = 2;
        else
            indent = (indent - 1) * 8;

        resultStr += indentStr(indent, str) + "\n"
    };
    output(0, "--- Sharding Status --- ");
    output(1, "sharding version: " + tojson(configDB.getCollection("version").findOne()));

    output(1, "shards:");
    configDB.shards.find().sort({ _id: 1 }).forEach(function (obj) {
        output(2, tojsononeline(obj));
    });

    var mongosActiveThresholdMs = 60000;
    var mostRecentMongos = configDB.mongos.find().sort({ ping: -1 }).limit(1);
    var mostRecentMongosTime = null;
    var mongosAdjective = "most recently active";
    if (mostRecentMongos.hasNext()) {
        mostRecentMongosTime = mostRecentMongos.next().ping;
        if (mostRecentMongosTime.getTime() >= Date.now() - mongosActiveThresholdMs) {
            mongosAdjective = "active";
        }
    }

    output(1, mongosAdjective + " mongoses:");
    if (mostRecentMongosTime === null) {
        output(2, "none");
    } else {
        var recentMongosQuery = {
            ping: {
                $gt: (function () {
                    var time = mostRecentMongosTime;
                    time.setTime(time.getTime() - mongosActiveThresholdMs);
                    return time;
                })()
            }
        };

        if (verbose) {
            configDB.mongos.find(recentMongosQuery).sort({ ping: -1 }).forEach(function (obj) {
                output(2, tojsononeline(obj));
            });
        } else {
            configDB.mongos
                .aggregate([
                    { $match: recentMongosQuery },
                    { $group: { _id: "$mongoVersion", num: { $sum: 1 } } },
                    { $sort: { num: -1 } }
                ])
                .forEach(function (obj) {
                    output(2, tojson(obj._id) + " : " + obj.num);
                });
        }
    }

    output(1, "autosplit:");

    output(2, "Currently enabled: " + (sh.getShouldAutoSplit(configDB) ? "yes" : "no"));

    output(1, "balancer:");

    output(2, "Currently enabled:  " + (sh.getBalancerState(configDB) ? "yes" : "no"));

    output(2, "Currently running:  " + (sh.isBalancerRunning(configDB) ? "yes" : "no"));

    var balSettings = sh.getBalancerWindow(configDB);
    if (balSettings)
        output(3, "Balancer active window is set between " + balSettings.start + " and " +
                      balSettings.stop + " server local time");

    var activeMigrations = sh.getActiveMigrations(configDB);
    if (activeMigrations.length > 0) {
        output(2, "Collections with active migrations: ");
        activeMigrations.forEach(function (migration) {
            output(3, migration._id + " started at " + migration.when);
        });
    }

    var versionHasActionlog = false;
    var metaDataVersion = configDB.getCollection("version").findOne().currentVersion;
    if (metaDataVersion > 5) {
        versionHasActionlog = true;
    }
    if (metaDataVersion == 5) {
        var verArray = this.serverBuildInfo().versionArray;
        if (verArray[0] == 2 && verArray[1] > 6) {
            versionHasActionlog = true;
        }
    }

    if (versionHasActionlog) {
        var actionReport = sh.getRecentFailedRounds(configDB);
        output(2, "Failed balancer rounds in last 5 attempts:  " + actionReport.count);

        if (actionReport.count > 0) {
            output(2, "Last reported error:  " + actionReport.lastErr);
            output(2, "Time of Reported error:  " + actionReport.lastTime);
        }

        output(2, "Migration Results for the last 24 hours: ");
        var migrations = sh.getRecentMigrations(configDB);
        if (migrations.length > 0) {
            migrations.forEach(function (obj) {
                if (obj._id === "Success") {
                    output(3, obj.count + " : " + obj._id);
                } else {
                    output(3, obj.count + " : Failed with error '" + obj._id + "', from " +
                           obj.from + " to " + obj.to);
                }
            });
        } else {
            output(3, "No recent migrations");
        }
    }

    output(1, "databases:");
    var databaseArray = configDB.databases.find().sort({name: 1}).toArray();

    databaseArray.push({ "_id": "config", "primary": "config", "partitioned": true });
    databaseArray.sort(function (a, b) {
        return a["_id"] > b["_id"];
    });

    databaseArray.forEach(function (database) {
        var truthy = function (value) {
            return !!value;
        };
        var nonBooleanNote = function (name, value) {
            var resultStr = "";
            if (!isBoolean(value) && !isUndefined(value)) {
                resultStr = " (" + name + ": " + tojson(value) + ")";
            }
            return resultStr;
        };

        output(2, tojsononeline(database, "", true));

        if (database.partitioned) {
            configDB.collections.find({ _id: new RegExp("^" + RegExp.escape(database._id) + "\\.") })
                .sort({ _id: 1 })
                .forEach(function (collection) {
                    if (!collection.dropped) {
                        output(3, collection._id);
                        output(4, "shard key: " + tojson(collection.key));
                        output(4, "unique: " + truthy(collection.unique) +
                               nonBooleanNote("unique", collection.unique));
                        output(4, "balancing: " + !truthy(collection.noBalance) +
                               nonBooleanNote("noBalance", collection.noBalance));
                        output(4, "chunks:");

                        var result = configDB.chunks
                                  .aggregate([
                                                { $match: { ns: collection._id } },
                                                { $group: { _id: "$shard", cnt: { $sum: 1 } } },
                                                { $project: { _id: 0, shard: "$_id", nChunks: "$cnt" } },
                                                { $sort: { shard: 1 } }
                                             ])
                                  .toArray();
                        var totalChunks = 0;
                        result.forEach(function (obj) {
                            totalChunks += obj.nChunks;
                            output(5, obj.shard + "\t" + obj.nChunks);
                        });

                        if (totalChunks < 20 || verbose) {
                            configDB.chunks.find({ "ns": collection._id })
                                .sort({ min: 1 })
                                .forEach(function (chunk) {
                                    output(4, tojson(chunk.min) + " -->> " +
                                           tojson(chunk.max) + " on : " + chunk.shard + " " +
                                           tojson(chunk.lastmod) + " " +
                                           (chunk.jumbo ? "jumbo " : ""));
                                });
                        } else {
                            output(4, "too many chunks to print, use verbose if you want to force print");
                        }

                        configDB.tags.find({ ns: collection._id }).sort({ min: 1 }).forEach(function (tag) {
                            output(4, "tag: " + tag.tag + "  " + tojson(tag.min) + " -->> " + tojson(tag.max));
                        });
                    }
                });
        }
    });

    print(resultStr);
};

NAVDatabase.prototype.printSlaveReplicationInfo = function () {
    var startOptimeDate = null;
    var primary = null;

    function getReplLag(st) {
        if (!startOptimeDate)
            nav_throwError("getReplLag startOptimeDate cannot be null");
        print("\tsyncedTo: " + st.toString() + "\n");
        var ago = (startOptimeDate - st) / 1000;
        var hours = Math.round(ago / 36) / 100;
        var suffix = "";
        if (primary)
            suffix = "primary ";
        else
            suffix = "freshest member (no primary available at the moment)";
        print("\t" + Math.round(ago) + " secs (" + hours + " hrs) behind the " + suffix + "\n");
    }

    function getMaster(members) {
        for (i in members) {
            var row = members[i];
            if (row.state === 1)
                return row;
        }

        return null;
    }

    function g(obj) {
        if (!obj)
            nav_throwError("printSlaveReplicationInfo g obj cannot be null");
        print("source: " + obj.host + "\n");
        if (obj.syncedTo) {
            var date = new Date(this.tsToSeconds(obj.syncedTo) * 1000);
            getReplLag(date);
        } else {
            print("\tdoing initial sync\n");
        }
    }

    function printNodeReplicationInfo(obj) {
        if (!obj)
            nav_throwError("printSlaveReplicationInfo r obj cannot be null");
        if (obj.state == 1 || obj.state == 7)
            return;

        print("source: " + obj.name + "\n");
        if (obj.optime)
            getReplLag(obj.optimeDate);
        else
            print("\tno replication info, yet.  State: " + obj.stateStr + "\n");
    }
    function printNodeInitialSyncInfo(syncSourceString, remainingMillis) {
        print("\tInitialSyncSyncSource: " + syncSourceString);
        let minutes = Math.floor((remainingMillis / (1000 * 60)) % 60);
        let hours = Math.floor(remainingMillis / (1000 * 60 * 60));
        print("\tInitialSyncRemainingEstimatedDuration: " + hours + " hour(s) " + minutes +
            " minute(s)");
    }

    var localDatabase = this.getSiblingDB("local");

    if (localDatabase.system.replset.count() != 0) {
        var status = this.adminCommand({ replSetGetStatus: 1 });
        primary = getMaster(status.members);
        if (primary) {
            startOptimeDate = primary.optimeDate;
        } else {
            startOptimeDate = new Date(0, 0);
            for (i in status.members) {
                if (status.members[i].optimeDate > startOptimeDate) {
                    startOptimeDate = status.members[i].optimeDate;
                }
            }
        }

        for (i in status.members) {
            if (status.members[i].self && status.members[i].state === 5) {
                print("source: " + status.members[i].name);
                if (!status.initialSyncStatus) {
                    print("InitialSyncStatus information not found");
                    continue;
                }

                printNodeInitialSyncInfo(
                    status.members[i].syncSourceHost,
                    status.initialSyncStatus.remainingInitialSyncEstimatedMillis);
            } else {
                printNodeReplicationInfo(status.members[i]);
            }
        }
    } else if (localDatabase.sources.count() != 0) {
        startOptimeDate = new Date();
        localDatabase.sources.find().forEach(g);
    } else {
        print("local.sources is empty; is this db a --slave?");
        return;
    }
};
NAVDatabase.prototype.printSecondaryReplicationInfo = NAVDatabase.prototype.printSlaveReplicationInfo;

NAVDatabase.prototype.repairDatabase = function () {
    var commandObj = { repairDatabase: 1 };
    return this.runCommand(commandObj);
};

NAVDatabase.prototype.resetError = function () {
    var commandObj = { reseterror: 1 };
    return this.runCommand(commandObj);
};

NAVDatabase.prototype.rotateCertificates = function (message) {
    var commandObj = { rotateCertificates: 1, message: message };
    return this.adminCommand(commandObj);
};

NAVDatabase.prototype.runCommand = function (command, options) {
    if (!isString(command) && !isObject(command))
        nav_throwError("command must be string or object");

    var readPreference = undefined;
    if (isObject(options)) {
        if (isString(command))
            command = { command: 1 };

        Object.extend(command, options);

        if (isString(command.readPreference)) {
            readPreference = command.readPreference;
            delete command.readPreference;
        }
    }

    // about AzureCosmosDB, we does not support {usersInfo:1 } (although can run normally in 3T and mongosh)
    // as a tmp fix, we replace it into {usersInfo:{} } 
    if (this.getMongo().forwardToCustomFunction("mongoCCGetValue", "isMongoAzureCosmosDB")) {
        if (command.usersInfo === 1) {
            command.usersInfo = {};
        }
    }

    return this.forwardToCustomFunction("databaseRunCommand", command, readPreference);
};

NAVDatabase.prototype.serverBuildInfo = function () {
    var commandObj = { buildinfo: 1 };
    return this.adminCommand(commandObj);
};

NAVDatabase.prototype.serverCmdLineOpts = function () {
    var commandObj = { getCmdLineOpts: 1 };
    return this.adminCommand(commandObj);
};

NAVDatabase.prototype.serverStatus = function (options) {
    var commandObj = { serverStatus: 1 };
    if (options)
        Object.extend(commandObj, options);
    return this.adminCommand(commandObj);
};

NAVDatabase.prototype.setLogLevel = function (level, component) {
    var componentNames = [];
    if (isString(component))
        componentNames = component.split(".");
    else if (!isUndefined(component))
        nav_throwError("setLogLevel component must be a string");

    var verbosityObj = { verbosity: level };
    while (componentNames.length)
        verbosityObj = { [componentNames.pop()]: verbosityObj };

    var commandObj = { setParameter: 1, logComponentVerbosity: verbosityObj };

    var result = this.adminCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.setProfilingLevel = function (level, options) {
    if (isUndefined(level))
        nav_throwError("Level should be defined");
    if (!isNumber(level))
        nav_throwError("Level should be number");
    if (level < 0 || level > 2)
        nav_throwError("Level " + level + " is out of range [0..2]");

    var commandObj = { profile: level };
    if (isNumber(options))
        commandObj.slowms = options;
    else
        commandObj = Object.extend(commandObj, options);

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.stats = function (options) {
    var command = { dbstats: 1 };

    if (isUndefined(options))
        return this.runCommand(command);
    if (!isObject(options))
        return this.runCommand(Object.extend(command, { scale: options }));

    return this.runCommand(Object.extend(command, options));
};

NAVDatabase.prototype.version = function () {
    return this.serverBuildInfo().version;
};

NAVDatabase.prototype.watch = function (pipeline, options) {
    var tmpPipeline = Object.extend([], (pipeline || []));
    var tmpOptions = Object.extend({}, (options || {}));
    if (!tmpPipeline instanceof Array)
        nav_throwError("pipeline should be array");
    if (!tmpOptions instanceof Object)
        nav_throwError("options should be object");

    var changeStreamStage = { fullDocument: tmpOptions.fullDocument || "default" };
    delete tmpOptions.fullDocument;

    if (tmpOptions.hasOwnProperty("resumeAfter")) {
        changeStreamStage.resumeAfter = tmpOptions.resumeAfter;
        delete tmpOptions.resumeAfter;
    }

    if (tmpOptions.hasOwnProperty("startAfter")) {
        changeStreamStage.startAfter = tmpOptions.startAfter;
        delete tmpOptions.startAfter;
    }

    if (tmpOptions.hasOwnProperty("fullDocumentBeforeChange")) {
        changeStreamStage.fullDocumentBeforeChange = tmpOptions.fullDocumentBeforeChange;
        delete tmpOptions.fullDocumentBeforeChange;
    }

    if (tmpOptions.hasOwnProperty("allChangesForCluster")) {
        changeStreamStage.allChangesForCluster = tmpOptions.allChangesForCluster;
        delete tmpOptions.allChangesForCluster;
    }

    if (tmpOptions.hasOwnProperty("allowToRunOnConfigDB")) {
        changeStreamStage.allowToRunOnConfigDB = tmpOptions.allowToRunOnConfigDB;
        delete tmpOptions.allowToRunOnConfigDB;
    }

    if (tmpOptions.hasOwnProperty("allowToRunOnSystemNS")) {
        changeStreamStage.allowToRunOnSystemNS = tmpOptions.allowToRunOnSystemNS;
        delete tmpOptions.allowToRunOnSystemNS;
    }

    if (tmpOptions.hasOwnProperty("startAtOperationTime")) {
        changeStreamStage.startAtOperationTime = tmpOptions.startAtOperationTime;
        delete tmpOptions.startAtOperationTime;
    }

    if (tmpOptions.hasOwnProperty("showExpandedEvents")) {
        changeStreamStage.showExpandedEvents = tmpOptions.showExpandedEvents;
        delete tmpOptions.showExpandedEvents;
    }

    if (tmpOptions.hasOwnProperty("showSystemEvents")) {
        changeStreamStage.showSystemEvents = tmpOptions.showSystemEvents;
        delete tmpOptions.showSystemEvents;
    }

    if (tmpOptions.hasOwnProperty("showRawUpdateDescription")) {
        changeStreamStage.showRawUpdateDescription = tmpOptions.showRawUpdateDescription;
        delete tmpOptions.showRawUpdateDescription;
    }

    tmpPipeline.unshift({ $changeStream: changeStreamStage });
    return this.aggregate(tmpPipeline, tmpOptions);
};

// Free Monitoring
NAVDatabase.prototype.getFreeMonitoringStatus = function() {
    var command = {getFreeMonitoringStatus: 1};
    return this.adminCommand(command);
};

NAVDatabase.prototype.enableFreeMonitoring = function() {
    var command = {setFreeMonitoring: 1, action: 'enable'};
    this.adminCommand(command);

    command = {getFreeMonitoringStatus: 1};
    var result = this.adminCommand(command);
    if (!result.ok && (result.code == ErrorCode.Unauthorized)) {
        // Edge case: It's technically possible that a user can change free-mon state,
        // but is not allowed to inspect it.
        print("Successfully initiated free monitoring, but unable to determine status " +
              "as you lack the 'checkFreeMonitoringStatus' privilege.");
        return;
    }

    if (result.state !== 'enabled') {
        print("Successfully initiated free monitoring. The registration is " +
              "proceeding in the background. ");
        print("Run db.getFreeMonitoringStatus() at any time to check on the progress.");
        return;
    }

    print(tojson(cmd));
};

NAVDatabase.prototype.disableFreeMonitoring = function() {
    var command = {setFreeMonitoring: 1, action: 'disable'};
    this.adminCommand(command);
};

// User Management
function _obtainWriteConcern(writeConcern) {
    if (writeConcern)
        return writeConcern;
    return { w: 'majority', wtimeout: 10 * 60 * 1000 }
}

NAVDatabase.prototype.changeUserPassword = function (userName, password, writeConcern) {
    if (isUndefined(userName))
        nav_throwError("User name should be specified");
    if (!isString(userName))
        nav_throwError("User name should be string");

    return this.updateUser(userName, { pwd: password }, writeConcern);
}

function _hashPassword(userName, password) {
    if (!isString(password))
        nav_throwError("User password should be string");

    return hex_md5(userName + ":mongo:" + password);
}

NAVDatabase.prototype._modifyCommandToDigestPasswordIfNecessary = function (commandObj, userName) {
    if (!commandObj.pwd)
        return;
    if (commandObj.passwordDigestor === undefined)
        return;
    if (commandObj.hasOwnProperty("digestPassword"))
        nav_throwError("Cannot set digestPassword, has to use passwordDigestor instead");

    var passwordDigestor = commandObj.passwordDigestor ? commandObj.passwordDigestor : "server";
    if (passwordDigestor == "server") {
        commandObj.digestPassword = true;
    } else if (passwordDigestor == "client") {
        commandObj.pwd = _hashPassword(userName, commandObj.pwd);
        commandObj.digestPassword = false;
    } else {
        nav_throwError("passwordDigestor = " + passwordDigestor + ", where passwordDigestor can only be either server or client");
    }
    delete commandObj.passwordDigestor;
};

NAVDatabase.prototype.createUser = function (userObj, writeConcern) {
    var userName = userObj.user;
    if (isUndefined(userName))
        nav_throwError("createUser should have user name");

    var commandObj = { createUser: userName };
    commandObj = Object.extend(commandObj, userObj);
    delete commandObj.user;

    this._modifyCommandToDigestPasswordIfNecessary(commandObj, userName);

    commandObj.writeConcern = _obtainWriteConcern(writeConcern);

    var result = this.runCommand(commandObj);

    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.dropUser = function (userName, writeConcern) {
    var commandObj = {
        dropUser: userName,
        writeConcern: _obtainWriteConcern(writeConcern)
    };

    var result = this.runCommand(commandObj)
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
}
NAVDatabase.prototype.removeUser = NAVDatabase.prototype.dropUser;

NAVDatabase.prototype.dropAllUsers = function (writeConcern) {
    var commandObj = {
        dropAllUsersFromDatabase: 1,
        writeConcern: _obtainWriteConcern(writeConcern)
    };

    var result = this.runCommand(commandObj)
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result.n;
}

NAVDatabase.prototype.getUser = function (userName, args) {
    if (!isString(userName))
        nav_throwError("User name should be string");

    var commandObj = { usersInfo: userName };
    Object.extend(commandObj, args);

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    if (result.users.length == 0)
        return null;
    return result.users[0];
};

NAVDatabase.prototype.getUsers = function (args) {
    var commandObj = { usersInfo: 1 };
    Object.extend(commandObj, args);

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result.users;
};

NAVDatabase.prototype.grantRolesToUser = function (userName, roles, writeConcern) {
    var commandObj = {
        grantRolesToUser: userName,
        roles: roles,
        writeConcern: _obtainWriteConcern(writeConcern)
    };

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.revokeRolesFromUser = function (userName, roles, writeConcern) {
    var commandObj = {
        revokeRolesFromUser: userName,
        roles: roles,
        writeConcern: _obtainWriteConcern(writeConcern)
    };

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.updateUser = function (userName, updateObject, writeConcern) {
    var commandObj = { updateUser: userName };
    commandObj = Object.extend(commandObj, updateObject);
    commandObj.writeConcern = _obtainWriteConcern(writeConcern);
    this._modifyCommandToDigestPasswordIfNecessary(commandObj, userName);

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

// Role Management
NAVDatabase.prototype.createRole = function (roleObj, writeConcern) {
    var roleName = roleObj.role;
    var commandObj = { createRole: roleName };
    commandObj = Object.extend(commandObj, roleObj);
    delete commandObj.role;
    commandObj.writeConcern = _obtainWriteConcern(writeConcern);

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.dropRole = function (roleName, writeConcern) {
    var commandObj = {
        dropRole: roleName,
        writeConcern: _obtainWriteConcern(writeConcern)
    };

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.dropAllRoles = function (writeConcern) {
    var commandObj = {
        dropAllRolesFromDatabase: 1,
        writeConcern: _obtainWriteConcern(writeConcern)
    };

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result.n;
};

NAVDatabase.prototype.getRole = function (roleName, args) {
    if (!isString(roleName))
        nav_throwError("Role name should be string");

    var commandObj = { rolesInfo: roleName };
    Object.extend(commandObj, args);

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    if (result.roles.length == 0)
        return null;
    return result.roles[0];
};

NAVDatabase.prototype.getRoles = function (args) {
    var command = { rolesInfo: 1 };
    Object.extend(command, args);

    var result = this.runCommand(command);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result.roles;
};

NAVDatabase.prototype.grantPrivilegesToRole = function (roleName, privileges, writeConcern) {
    var commandObj = {
        grantPrivilegesToRole: roleName,
        privileges: privileges,
        writeConcern: _obtainWriteConcern(writeConcern)
    };

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.revokePrivilegesFromRole = function (roleName, privileges, writeConcern) {
    var commandObj = {
        revokePrivilegesFromRole: roleName,
        privileges: privileges,
        writeConcern: _obtainWriteConcern(writeConcern)
    };

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.grantRolesToRole = function (roleName, roles, writeConcern) {
    var commandObj = {
        grantRolesToRole: roleName,
        roles: roles,
        writeConcern: _obtainWriteConcern(writeConcern)
    };

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.revokeRolesFromRole = function (roleName, roles, writeConcern) {
    var commandObj = {
        revokeRolesFromRole: roleName,
        roles: roles,
        writeConcern: _obtainWriteConcern(writeConcern)
    };

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.updateRole = function (roleName, updateObject, writeConcern) {
    var commandObj = { updateRole: roleName };
    commandObj = Object.extend(commandObj, updateObject);
    commandObj.writeConcern = _obtainWriteConcern(writeConcern);

    var result = this.runCommand(commandObj);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVDatabase.prototype.checkMetadataConsistency = function (options = {}) {
    if (!isObject(options))
        nav_throwError("'options' parameter is " + typeof options + " but not object");

    var result = this.runCommand(Object.extend({ checkMetadataConsistency: 1 }, options));
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result.cursor.firstBatch;
};
